function el = rv2el(r,v,mu)

% 函数功能：
%         将位置矢量和速度矢量转化为经典轨道根数。


% 输入参数：
% r  -- 位置矢量，单位为km
% v  -- 速度矢量，单位为km/s 
% mu -- 中心天体引力常数，km^3/s^2 

% 输出参数：
% el -- 经典轨道根数[sma ecc incl OM om theta]
% sma   - 半长轴，单位：km
% ecc   - 偏心率
% incl  - 倾角，单位：弧度
% OM    - 升交点经度，单位：弧度
% om    - 近地点幅角，单位：弧度
% theta - 真近点角，单位：弧度

% 定义无量纲单位
AU  = sqrt(r(1)*r(1)+r(2)*r(2)+r(3)*r(3));
VU  = sqrt(mu/AU);

% 误差精度
eps = 1e-14; 

% 飞行器的无量纲位置矢量和速度矢量，及其夹角
R = r/AU;
V = v/VU;
nv = norm(V); 
csy = acos(dot(R,V)/nv);

% 无量纲径向速度和切向速度
vr = nv*cos(csy);
vt = nv*sin(csy);

% 半长轴，偏心率和真近点角
sma = 1/(2 - nv*nv);
esinf = vr*vt;
ecosf = vt*vt - 1;
ecc = sqrt(esinf*esinf + ecosf*ecosf);
theta = mod(atan2(esinf,ecosf),2*pi);

H = cross(R,V)/vt; % 单位法向方向
si = sign(H(3)); % 顺行1，逆行-1

if H(3)>1-eps
   N = [1 0 0]'; %升交点单位矢量
   incl = 0;
   OM = 0;
elseif H(3)<eps-1
   N = [1 0 0]'; %升交点矢量
   incl = pi;
   OM = 0;
else
   incl = acos(H(3));
   N = [-H(2) H(1) 0]'; N = N/norm(N);%升交点单位矢量
   OM = mod(atan2(N(2),N(1)),2*pi);   
end

NcR = cross(N,R);
NdR = dot(N,R);
if abs(NdR)>1-eps
     fandom = (1-sign(NdR))*pi/2;  
else
     fandom = acos(NdR);
     if si*NcR(3)<0
        fandom = 2*pi-fandom;
     end
end
om = mod(fandom-theta,2*pi);

el = [sma*AU ecc incl OM om theta];


end
